/*
*  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
*
*  This file is part of the Arnold emulator source code distribution.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include "cpc.h"
#include "pal.h"
#include "emudevice.h"
#include "memrange.h"
#include "riff.h"

typedef struct
{
	unsigned char *Ram;
	int RamConfig;

	BOOL ActiveSwitch;
	BOOL Switch64KB;
	BOOL BatterySwitch;
} InicronRam;

static InicronRam inicron;

BOOL  InicronRam_GetBattery(void)
{
	return inicron.BatterySwitch;
}

void  InicronRam_SetBattery(BOOL bState)
{
	inicron.BatterySwitch = bState;
}

BOOL InicronRam_GetS2(void)
{
	return inicron.ActiveSwitch;
}

void InicronRam_SetS2(BOOL bState)
{
	inicron.ActiveSwitch= bState;
}


BOOL InicronRam_GetS1(void)
{
	return inicron.Switch64KB;
}

void InicronRam_SetS1(BOOL bState)
{
	inicron.Switch64KB = bState;
}


void InicronRam_Reset(void)
{
	/* TODO: Confirm reset/power on */
	inicron.RamConfig = 0;

}

void InicronRam_MemoryRethink(MemoryData *pData)
{
	/* TODO: Confirm RAMDIS is recognised */
	if (inicron.ActiveSwitch&& (inicron.RamConfig & (1<<2)))
	{
		switch (inicron.RamConfig&0x03)
		{
		case 0:
		case 1:
		case 2:
		case 3:
		{
			int Page = inicron.RamConfig & 0x03;
			int Bank = (inicron.RamConfig>>3)&0x07;

			/* inicron 64kb not active and page 0 selected */
			if (!inicron.Switch64KB && (Bank==0))
				break;

			/* 4000-7fff only */
			pData->pWritePtr[2] = ((inicron.Ram + (Bank << 16)) + ((Page << 14)) - 0x04000);
			pData->pWritePtr[3] = pData->pWritePtr[2];

			pData->pReadPtr[2] = pData->pWritePtr[2];
			pData->pReadPtr[3] = pData->pWritePtr[3];
			/* disable internal ram */
			pData->bRamDisable[2] = TRUE;
			pData->bRamDisable[3] = TRUE;
			pData->pReadMaskPtr[3] = GetDefaultReadMask() - 0x04000;
			pData->pReadMaskPtr[2] = GetDefaultReadMask() - 0x04000;
		}
		break;
		default:
		break;
		}
	}
}

static void InicronRam_PortWrite(Z80_WORD Port, Z80_BYTE Data)
{
	/* TODO: Confirm bit 7,6 decoding */
    if ((Data & 0x0c0)==0x0c0)
	{
		inicron.RamConfig = Data;

		Computer_RethinkMemory();
	}
}



static EmuDeviceSwitch InicronRamSwitches[3]=
{
  {
      "S1 - Override extra 64K RAM (CPC6128)",
	  "S1",
      InicronRam_GetS1,
      InicronRam_SetS1
  },
  {
      "S2 - Activate",
	  "S2",
      InicronRam_GetS2,
      InicronRam_SetS2
  },
   {
      "Battery jumper",
	  "BatteryJumper",
      InicronRam_GetBattery,
      InicronRam_SetBattery
  }
};

/* TODO: Confirm port decoding */
CPCPortWrite InicronRamPortWrite[1]=
{
	{
		0x0ff00,
		0x07f00,
		InicronRam_PortWrite
	}
};


void InicronRamDevice_Init(void)
{
	inicron.ActiveSwitch = TRUE;
	inicron.BatterySwitch = FALSE;
	inicron.Switch64KB = TRUE;
	inicron.RamConfig = 0;
	inicron.Ram = (unsigned char *)malloc(512 * 1024);
	memset(inicron.Ram, 0x0ff, 512 * 1024);

}

void InicronRamDevice_Shutdown(void)
{
	if (inicron.Ram)
	{
		free(inicron.Ram);
	}
}

static EmuDevice InicronRamDevice =
{
	NULL,
	InicronRamDevice_Init,
	InicronRamDevice_Shutdown,
	"InicronRam",
	"InicronRam",
	"InicronRam (External 512KB RAM)",
	CONNECTION_EXPANSION,   /* connects to expansion */
	DEVICE_WORKING,	/* has ram but not fully dk'tronics compatible */
	0,
	NULL,					/* no read ports */
	sizeof(InicronRamPortWrite) / sizeof(InicronRamPortWrite[0]),
	InicronRamPortWrite,			/* 1 write port */
	0,                /* no memory read*/
	NULL,
	0,                /* no memory write */
	NULL,
	InicronRam_Reset,
	InicronRam_MemoryRethink,
	InicronRam_Reset,
	sizeof(InicronRamSwitches)/sizeof(InicronRamSwitches[0]),                      /* no switches */
	InicronRamSwitches,
	0,                      /* 2 buttons */
	NULL,
	0,                      /* 1 onboard roms */
	NULL,
	0,
	NULL,
	NULL,                   /* no cursor function */
	NULL,                   /* no generic roms */
	NULL,					/* printer */
	NULL,					/* joystick */
	0,
	NULL,					/* memory ranges */
	NULL, /* sound */
	NULL, /* lpen */
	NULL, /* reti */
	NULL, /* ack maskable interrupt */
	NULL, /* dkram data */
	NULL, /* device ram */
	NULL, /* device backup */
	NULL,
};

int InicronRAM_Init()
{
	return RegisterDevice(&InicronRamDevice);
}

